﻿// #region ArcGisMapPlugin

(function ()
{
	var isDefined = false;

	function definePluginClass(settings)
	{
		if (isDefined)
			return;

		PP.ArcGisMapPlugin = function (settings)
		{
			PP.extend(settings, this._ExternalSettings, true, true);
			PP.ArcGisMapPlugin.base.constructor.call(this, settings);

			//Ассоциативный массив графических объектов esri/graphic по идентификатору шейпа карты
			//Графические объекты храянтся в формате { <shapeId>: [<Grapic>, <...>, ...], ... }
			this._aGraphics = {};
			//Кэш границ шейпов
			this._shapeBoundsCache = {};
			//Хранение стиля заливки полигонов по идентификатору шейпа
			this._fillSymbols = {};
		};

		PP.initClass(PP.ArcGisMapPlugin, PP.ExternalMapPlugin, 'PP.ArcGisMapPlugin');

		var amp = PP.ArcGisMapPlugin.prototype;
		amp._ExternalSettings = settings;

		amp._Basemap = 'topo'; //базовая карта для отображения тайлов. ArcGis предоставляет	несколько возможных вариантов базовой карты. Подробнее http://arcg.is/1JVo6Wd

		amp._render = function ()
		{
			//размещать карту нужно в ифрэйме, т.к. API аркгиса экспортирует большое количество глобальных имен, которые могут конфликтовать с API других карт (например Bing)
			this._initFromHTML('<div DraggableArea="true" style="position:relative;">\
				<iframe src="about:blank" style="width:100%; height:100%; border: none; padding: 0px; margin: 0px;"></iframe>\
		</div>');

			this._domReady = new Promise(function (resolve, reject)
			{
				this._DomNode.firstElementChild.onload = resolve;
			}.bind(this));

			this._domReady.then(function ()
			{
				this._frameContext = this._DomNode.firstElementChild.contentWindow;
				this._mapNode = this._frameContext.document.body;
				this._scriptsNode = this._frameContext.document.head;

				this._mapNode.setAttribute('style', 'width: 100%; height: 100%; padding: 0px; margin: 0px;');
			}.bind(this));
		};

		amp._initExternalAPI = function ()
		{
			this._domReady.then(function ()
			{
				var arcgisJsUrl = 'https://js.arcgis.com/3.20/',
				arcgisCssUrl = 'https://js.arcgis.com/3.20/esri/css/esri.css';

				PP.ScriptManager.current.setParentNode(this._scriptsNode);
				PP.ScriptManager.loadStyle(arcgisCssUrl);
				if (!PP.ScriptManager.loadScript(arcgisJsUrl, this._onArcGisApiLoaded.bind(this)))
					this._onArcGisApiLoaded();
				PP.ScriptManager.current.setParentNode(null);
			}.bind(this));
		};

		amp._onArcGisApiLoaded = function ()
		{
			//ArcGIS загружает dojo-toolkit с возможностью загрузки модулей посредствои реализации AMD внутри dojo

			//сразу загружаем все модули, которые нам потребуются при работе
			this._frameContext.require([
				'esri/map', //модуль для создания самого объекат карты
				'esri/graphic', //для рисования на карте графических объектов
				'esri/symbols/SimpleFillSymbol', // для задания заливки
				'esri/symbols/SimpleLineSymbol', // для задания обводки
				'esri/geometry/Polygon', // необходимая геометрия для графических объектов		
				'esri/Color', //для работы с цветом
				'esri/geometry/Point', //для работы с координатами на карте
				'esri/layers/GraphicsLayer', //Для отображения внешних объектов на карте (стрелки, маркеры, графики и т.п.)
				'esri/graphicsUtils' //Для вычисления границ полигонов
			], function (Map, Graphic, SimpleFillSymbol, SimpleLineSymbol, Polygon, Color, Point, GraphicsLayer, graphicsUtils)
			{
				//скидываем ссылки на модули во внутреннюю переменную класса
				this._api = {
					Map: Map,
					Graphic: Graphic,
					SimpleFillSymbol: SimpleFillSymbol,
					SimpleLineSymbol: SimpleLineSymbol,
					Polygon: Polygon,
					Color: Color,
					Point: Point,
					GraphicsLayer: GraphicsLayer,
					graphicsUtils: graphicsUtils
				};

				this._apiReady = true;
			}.bind(this));
		};

		amp._createMap = function ()
		{
			if (!this._aMap)
			{
				this._aMap = new this._api.Map(this._mapNode, {
					basemap: this._Basemap,
					zoom: 2,
					center: this._getCenter().reverse()
				});

				//Слой для рисования шейпов
				this._shapeLayer = new this._api.GraphicsLayer();
				//Слой для специальных объектов карты (стрелки, графики, маркеры и т.п.)
				this._customLayer = new this._api.GraphicsLayer();

				this._aMap.on('load', this._onMapLoaded.bind(this));
			} else
			{
				//нужно снести все нарисованные полигоны т.к. пришла новая топооснова
				this._shapeLayer.clear();
				this._aGraphics = {};
				this._fillSymbols = {};
				this._shapeBoundsCache = {};

				this._isReady = true;
				this.Ready.fire(this);
			}
		};

		amp._onMapLoaded = function ()
		{
			if (this.isLive())
			{
				this._aMap.on('zoom-end', this._onZoomEnd.bind(this));

				//Подписываемся на события слоя с шейпами
				this._shapeLayer.on('click', this._onPolygonMouseClick.bind(this));
				this._shapeLayer.on('mouse-move', this._onPolygonMouseMove.bind(this));
				this._shapeLayer.on('mouse-over', this._onPolygonMouseOver.bind(this));
				this._shapeLayer.on('mouse-out', this._onPolygonMouseOut.bind(this));

				this._aMap.addLayer(this._shapeLayer);
				this._aMap.addLayer(this._customLayer);

				this._isReady = true;
				this.Ready.fire(this);
			}
		};

		amp._onZoomEnd = function ()
		{
			if (this._isReady)
				this.Refreshed.fire(this);
		};

		amp._fixEvent = function (e)
		{
			//т.к. карта находится в iframe координаты pageX и pageY эвента приходят относительно фрейма
			//прибавим к ним смещение left, top ноды плагина

		    var res = {
		        originalEvent: e
		    },
				domNodeRect = this._DomNode.getBoundingClientRect();

		    for (var p in e)
		    {
		        res[p] = e[p];
		    }

			res.pageX += domNodeRect.left;
			res.pageY += domNodeRect.top;

			return res;
		};

		amp._onPolygonMouseOver = function (e)
		{
			if (!e.graphic || !e.graphic.__ppShape || !e.graphic.__ppShape.isLive())
				return;

			this.ShapeMouseOver.fire(this, { Shape: e.graphic.__ppShape, Event: this._fixEvent(e) });
		};

		amp._onPolygonMouseMove = function (e)
		{
			if (!e.graphic || !e.graphic.__ppShape || !e.graphic.__ppShape.isLive())
				return;

			this.ShapeMouseMove.fire(this, { Shape: e.graphic.__ppShape, Event: this._fixEvent(e) });
		};

		amp._onPolygonMouseOut = function (e)
		{
			if (!e.graphic || !e.graphic.__ppShape || !e.graphic.__ppShape.isLive())
				return;

			this.ShapeMouseOut.fire(this, { Shape: e.graphic.__ppShape, Event: this._fixEvent(e) });
		};

		amp._onPolygonMouseClick = function (e)
		{
			if (!e.graphic || !e.graphic.__ppShape || !e.graphic.__ppShape.isLive())
				return;

			this.ShapeClick.fire(this, { Shape: e.graphic.__ppShape, Event: this._fixEvent(e) });
		};

		amp.setSize = function (width, height)
		{
			PP.ArcGisMapPlugin.base.setSize.apply(this, arguments);

			if (this._aMap)
			{
				this._aMap.resize();
			}
		};

		amp.drawShape = function (shape, isVisible, isStroked, isGradient, color, opacity, borderColor, borderWidth)
		{
			if (!this._isReady || !shape || !shape.isLive())
				return;

			var shapeId = shape.getId(),
				data = shape.getPluginData(),
				i;

			if (!this._aGraphics[shapeId])
			{
				if (!isVisible || !data.polygons.length)
					return;

				this._aGraphics[shapeId] = [];
				//1. Создаем заливку и обводку. Она будет одинакова для всех частей шейпа
				var fillColor = new this._api.Color(color);

				fillColor.a = 0.85 * opacity;
				this._fillSymbols[shapeId] = new this._api.SimpleFillSymbol(
					isStroked ? this._api.SimpleFillSymbol.STYLE_NULL : this._api.SimpleFillSymbol.STYLE_SOLID,
					new this._api.SimpleLineSymbol(this._api.SimpleLineSymbol.STYLE_SOLID), new this._api.Color(borderColor), isStroked ? borderWidth : 0,
					fillColor);

				var normPolygons = this._normalisePolygons(data.polygons);

				for (i = 0; i < normPolygons.length; i++)
				{
					//2. Создаем геометрию			
					var coords = normPolygons[i];

					//3. Создаем графический объект и добавляем его в карту и во внутренний кэш
					var graphic = new this._api.Graphic(new this._api.Polygon(coords), this._fillSymbols[shapeId]);

					graphic.__ppShape = shape;

					this._aGraphics[shapeId].push(graphic);
					this._shapeLayer.add(graphic);
				}

			} else
			{
				//Меняем настройки соответствующей заливки полигонов и перерисовываем 
				if (isVisible)
				{
					this._fillSymbols[shapeId].setStyle(isStroked ? this._api.SimpleFillSymbol.STYLE_NULL : this._api.SimpleFillSymbol.STYLE_SOLID);
					this._fillSymbols[shapeId].color.setColor(color);
					this._fillSymbols[shapeId].color.a = 0.85 * opacity;
					this._fillSymbols[shapeId].outline.color.setColor(borderColor);
					this._fillSymbols[shapeId].outline.width = isStroked ? borderWidth : 0;
				}

				for (i = 0; i < this._aGraphics[shapeId].length; i++)
				{
					if (isVisible)
						this._aGraphics[shapeId][i].draw();
					else
						this._shapeLayer.remove(this._aGraphics[shapeId][i]);
				}

				if (!isVisible)
				{
					delete this._aGraphics[shapeId];
					delete this._fillSymbols[shapeId];
					delete this._shapeBoundsCache[shapeId];
				}
			}
		};

		amp._normalisePolygons = function (polygons)
		{
			//Метод проверяет есть ли среди полигонов шэйпа такие, в которых точки расположены на расстоянии более 180 градусов по долготе друг от друга
			//Если такие имеются, то выполняется нормализация координат всех полигонов шейпа в сторону "восточных" (положительных координат)
			//Это необходимо для правильного построения шейпов API ArcGis и корректного вычисления центра шейпов

			var eastPoly = [], //полигоны в восточных координатах
				westPoly = [], //полигоны в западных координатах
				centers = [], //центры полигонов
				isHasFarPoly = false, //признак, что в шейпе есть полигоны центры которых в более чем 180 градусах по долготе
				i, j,
				res = [];

			for (i = 0; i < polygons.length; i++)
			{
				var points = polygons[i].map(function (latlng)
				{
					//в аркгис координаты задаются в порядке lng, lat
					return latlng.slice().reverse();
				});

				var minLng = points[0][0],
					maxLng = points[0][0],
					isEast = true,
					isWest = true;

				for (j = 0; j < points.length; j++)
				{
					var p1 = points[j],
						p2 = j < points.length - 1 ? points[j + 1] : p1;

					if (Math.abs(p1[0] - p2[0]) > 180)
					{
						if (p1[0] >= 0)
							p2[0] += 360;
						else
							p2[0] -= 360;
					}

					if (p1[0] < 0)
						isEast = false;
					else
						isWest = false;

					if (p1[0] > maxLng)
						maxLng = p1[0];
					if (p1[0] < minLng)
						minLng = p1[0];
				}

				if (isEast)
					eastPoly.push(points);
				if (isWest)
					westPoly.push(points);

				centers.push((maxLng + minLng) / 2);
				res.push(points);
			}

			for (i = 0; i < centers.length && !isHasFarPoly; i++)
			{
				for (j = i + 1; j < centers.length; j++)
				{
					if (Math.abs(centers[i] - centers[j]) > 180)
					{
						isHasFarPoly = true;
						break;
					}
				}
			}

			if (isHasFarPoly && eastPoly.length && westPoly.length)
			{
				//нормализуем отрицательные координаты, так чтобы отсчитывать их к востоку от центрального меридиана
				//необходимо чтобы аркгис корректно считал экстент шейпа (массива полигонов)
				for (i = 0; i < westPoly.length; i++)
				{
					westPoly[i].forEach(function (point)
					{
						point[0] += 360;
					});
				}
			}

			return res;
		};

		amp.setShapeAttributes = function (shape, attributes)
		{
			/// <summary> Метод установки произвольных атрибутов шейпа. Имена атрибутов как в свг. </summary>
			/// <param name="shape" type="PP.Ui.MapShape"> Шейп. </param>
			/// <param name="attributes" type="Object"> Ассоциативный массив атрибутов. </param>

			if (!shape || !shape.isLive())
				return;

			var shapeId = shape.getId(),
				fillSymbol = this._fillSymbols[shapeId];

			if (!fillSymbol)
				return;

			for (var attr in attributes)
			{
				var value = attributes[attr];
				switch (attr)
				{
					case 'stroke':
						fillSymbol.outline.color.setColor(value);
						break;
					case 'stroke-opacity':
						fillSymbol.outline.color.a = value;
						break;
					case 'stroke-width':
						fillSymbol.outline.width = value;
						break;
					case 'color':
						fillSymbol.color.setColor(value);
						break;
				}
			}

			for (var i = 0; i < this._aGraphics[shapeId].length; i++)
				this._aGraphics[shapeId][i].draw();
		};

		amp.getShapeAttributes = function (shape, attributes)
		{
			/// <summary> Метод получения произвольных атрибутов шейпа. Имена атрибутов как в свг. </summary>
			/// <param name="shape" type="PP.Ui.MapShape"> Шейп. </param>
			/// <param name="attributes" type="String[]"> Массив имен атрибутов. </param>
			/// <returns type="Object"> Ассоциативный массив значений. </returns>

			return {};
		};

		amp.getShapeBoundsRect = function (shape)
		{
			/// <summary> Метод получения границ шейпа. </summary>
			/// <returns type="PP.Rect"> Границы. </returns>

			var shapeId = shape && shape.isLive() ? shape.getId() : null,
				resExtent = null;

			if (shapeId != null && this._aGraphics[shapeId])
			{
				if (this._shapeBoundsCache[shapeId])
					resExtent = this._shapeBoundsCache[shapeId];
				else
				{
					resExtent = this._api.graphicsUtils.graphicsExtent(this._aGraphics[shapeId]);
					this._shapeBoundsCache[shapeId] = resExtent;
				}
			}

			return new PP.Rect({
				Left: resExtent ? resExtent.getCenter().x : 0,
				Top: resExtent ? resExtent.getCenter().y : 0,
				Width: 0,
				Height: 0
			});
		};

		amp.geoToScreen = function (lng, lat)
		{
			if (this._isReady)
			{
				var aGeoPoint = new this._api.Point(lng, lat),
					aScrPoint = this._aMap.toScreen(aGeoPoint);

				return new PP.Point(aScrPoint.x, aScrPoint.y);
			}

			return null;
		};

		amp.getArrowLayer = amp.getMilestoneLayer =
		amp.getBarLayer = amp.getPieLayer =
		amp.getBubbleLayer = amp.getMarkerLayer = function ()
		{
			return this._isReady ? this._customLayer.getNode() : null;
		};

		amp.dispose = function ()
		{
			if (this._aMap)
				this._aMap.destroy();

			PP.ArcGisMapPlugin.base.dispose.apply(this, arguments);
		};

		amp = null;

		isDefined = true;
	}

	PP.ArcGis = function (settings)
	{
		PP.ArcGis.base.constructor.apply(this, arguments);
		this._Interfaces = [PP.IMapPlugin];
	};
	PP.initClass(PP.ArcGis, PP.Object, 'PP.ArcGis', [PP.IPlugin]);
	PP.Object.defineProps(PP.ArcGis, ['Interfaces'], false);

	PP.ArcGis.prototype._onReady = function (sender, args)
	{
		if (this._resolve)
			this._resolve();
		definePluginClass(args.Args);
		this._resolve = null;
	};

	PP.ArcGis.prototype.getInstance = function ()
	{
		return { 'ResourceKey': 'eArcGis', 'PPType': 'PP.ArcGisMapPlugin', 'TopobaseType': 'Google', 'ServiceTopobaseType': 'Google' };
	};

	PP.ArcGis.prototype.ready = function (resolve, reject)
	{
		if (PP.ExternalMapPlugin)
			return true;
		this._resolve = resolve;
		return false;
	};

	PP.ArcGis.init = function (settings)
	{
		if (PP.ExternalMapPlugin || !PP.ScriptManager.loadScript('ExternalMapPlugin.js', PP.Delegate(this._onReady, this, settings)))
			definePluginClass(settings);
		return new PP.ArcGis;
	};

	PP.OpenStreetMap = function (settings)
	{
		PP.OpenStreetMap.base.constructor.apply(this, arguments);
		this._Interfaces = [PP.IMapPlugin];
	};
	PP.initClass(PP.OpenStreetMap, PP.Object, 'PP.OpenStreetMap', [PP.IPlugin]);
	PP.Object.defineProps(PP.OpenStreetMap, ['Interfaces'], false);

	PP.OpenStreetMap.prototype._onReady = function (sender, args)
	{
		if (this._resolve)
			this._resolve();
		definePluginClass(args.Args);
		this._resolve = null;
	};

	PP.OpenStreetMap.prototype.getInstance = function ()
	{
		return { 'ResourceKey': 'eOSM', 'PPType': 'PP.ArcGisMapPlugin', 'Basemap': 'osm', 'TopobaseType': 'Google', 'ServiceTopobaseType': 'Google' };
	};

	PP.OpenStreetMap.prototype.ready = function (resolve, reject)
	{
		if (PP.ExternalMapPlugin)
			return true;
		this._resolve = resolve;
		return false;
	};

	PP.OpenStreetMap.init = function (settings)
	{
		if (PP.ExternalMapPlugin || !PP.ScriptManager.loadScript('ExternalMapPlugin.js', PP.Delegate(this._onReady, this, settings)))
			definePluginClass(settings);
		return new PP.OpenStreetMap;
	};
})();

// #endregion