#!/bin/bash

set -e

# Объявление директории, где необходимо размещать файлы бэкапа
if [ "$1" ]; then
  DATA_FOLDER=${1%/}
else
  DATA_FOLDER=$(find $(cd "$(dirname "${BASH_SOURCE[0]}")"; pwd -P) -maxdepth 1 -name "standalone_backup_*")
  array=(${DATA_FOLDER}) && n=${#array[@]}
  if [ "$n" -gt 1 ]; then
    echo -e "Укажите конкретный каталог с бэкапом: \n${DATA_FOLDER}" && exit 1
  elif [ "${n}" -lt 1 ]; then
    echo -e "Каталог с бэкапом не найден" && exit 1
  fi
fi

echo ${DATA_FOLDER}
COMPOSE_PROJECT_NAME="${2:-${COMPOSE_PROJECT_NAME:-fmp}}"

if echo $(docker ps --format "{{.Names}}") | grep -q "${COMPOSE_PROJECT_NAME}_hhive_"; then
    DJANGO_CONTAINER_NAME="hhive"
else
    DJANGO_CONTAINER_NAME="server"
fi
echo -e "Будем использовать django контейнер ${DJANGO_CONTAINER_NAME}"

function restore_postgres_data(){
  # Перечисляем БД для снятия бэкапа
  for db_label in main cache local
  do
    # Создаём команду для восстановления бэкапа в БД
    DB_SHELL_CMD="PGPASSWORD=\$POSTGRES_${db_label^^}_PASSWORD psql -h \$POSTGRES_${db_label^^}_HOST -p \$POSTGRES_${db_label^^}_PORT -d \$POSTGRES_${db_label^^}_NAME -U \$POSTGRES_${db_label^^}_USERNAME"

    echo -e "Чистим ${db_label}-базу от старых данных..."
    docker-compose -f docker-compose.standalone.yml exec -T ${DJANGO_CONTAINER_NAME} bash -c "${DB_SHELL_CMD} -c 'DROP SCHEMA public CASCADE; CREATE SCHEMA public;'"

    echo -e "Загружаем бэкап ${db_label}-базы из ${DATA_FOLDER}/${db_label}db.sql..."
    docker-compose -f docker-compose.standalone.yml exec -T ${DJANGO_CONTAINER_NAME} bash -c "${DB_SHELL_CMD}" < ${DATA_FOLDER}/${db_label}db.sql 2> /dev/null
  done
}

function restore_minio_data(){
  echo -e "Копируем бэкап в контейнер minio-базы из ${DATA_FOLDER}/minio_backup"
  MINIO_CONTAINER_NAME=$(docker ps --format "{{.Names}}" | grep "${COMPOSE_PROJECT_NAME}_minio")
  docker cp ${DATA_FOLDER}/minio_backup/ ${MINIO_CONTAINER_NAME}:/tmp/minio_backup

  echo "Загружаем бэкап в minio-базу..."
  RESTORE_COMMAND="mc mirror --overwrite --remove /tmp/minio_backup/ local"
  docker-compose -f docker-compose.standalone.yml exec -u root minio bash -c "${RESTORE_COMMAND}"

  echo "Удаляем бэкап minio-базы из контейнера"
  docker-compose -f docker-compose.standalone.yml exec -u root minio rm -fR /tmp/minio_backup
}

function restore_elasticsearch_data(){
  python3 elasticsearch_backup.py restore ${DATA_FOLDER}
}

function clear_redis_data(){
  echo "Сбрасываем кэш в редисе..."
  CLEAR_COMMAND="redis-cli --no-auth-warning -a \${REDIS_PASSWORD} -n 0 -c FLUSHDB"
  docker-compose -f docker-compose.standalone.yml exec redis bash -c "${CLEAR_COMMAND}"
}

function run_init_script(){
  # Здесь мы будем запускать команду init контейнера, чтобы накатить недостающие данные при обновлении ФМП
  echo "Запускаем init команду..."
  docker-compose -f docker-compose.standalone.yml exec -T ${DJANGO_CONTAINER_NAME} init.sh
}

restore_postgres_data
restore_elasticsearch_data
restore_minio_data
clear_redis_data
run_init_script
echo "Резервное восстановление завершено"