#!/bin/bash


set -e

DOCKER_COMPOSE_DIRECTORY_PATH="${1:-${DOCKER_COMPOSE_DIRECTORY_PATH:-$(pwd)}}"
COMPOSE_PROJECT_NAME="${2:-${COMPOSE_PROJECT_NAME:-fmp}}"
DATA_FOLDER="${3:-${DATA_FOLDER:-$(pwd)}}"


if echo $(docker ps --format "{{.Names}}") | grep -q "${COMPOSE_PROJECT_NAME}_hhive_"; then
    DJANGO_CONTAINER_NAME="hhive"
else
    DJANGO_CONTAINER_NAME="server"
fi

function wait_for_elastic_to_up() {
  echo -e "Ожидаем ответа от контейнера Elasticsearch "
  docker-compose -f docker-compose.standalone.yml exec -T ${DJANGO_CONTAINER_NAME} wait_for_elasticsearch.sh
  echo -e "Получен ответ от контейнера Elasticsearch"
}

function restore_elasticsearch_data(){
  url="\${ELASTICSEARCH_HOST}:\${ELASTICSEARCH_PORT}/_snapshot/user_script_backup_repository"
  creds="-u \${ELASTICSEARCH_USERNAME:-elastic}:\${ELASTICSEARCH_PASSWORD}"
  backup_path="/usr/share/elasticsearch/data/user_script_backup"
  exec_cmd="docker-compose -f ${DOCKER_COMPOSE_DIRECTORY_PATH}/docker-compose.standalone.yml exec -T ${DJANGO_CONTAINER_NAME} bash -c"

  echo -e "Копируем бэкап в контейнер.."
  ELASTIC_CONTAINER_NAME=$(docker ps --format "{{.Names}}" | grep "${COMPOSE_PROJECT_NAME}_elasticsearch")
  docker cp ${DATA_FOLDER}/elastic_backup/ ${ELASTIC_CONTAINER_NAME}:${backup_path}
  docker-compose -f ${DOCKER_COMPOSE_DIRECTORY_PATH}/docker-compose.standalone.yml exec -T -u root elasticsearch bash -c "chown elasticsearch:root -R ${backup_path}"


  echo -e "Создаём репозиторий.."
  put_repo_cmd="curl -X PUT -s ${creds} ${url} \
    -H 'Content-Type: application/json' \
    -d '{
      \"type\": \"fs\",
      \"settings\": {
        \"location\": \"${backup_path}\",
        \"compress\": true
      }
    }'"
  ${exec_cmd} "${put_repo_cmd}" > /dev/null

  echo -e "Очищаем индексы для установки бэкапа"
  # Предварительное удаление индексов ФМП чтобы они не мешали индексам из бэкапа.
  indices=$(${exec_cmd} "curl -s -X GET ${creds} \${ELASTICSEARCH_HOST}:\${ELASTICSEARCH_PORT}/_cat/indices/admin_log-*,user_log-*,mobile_log-*,periodic_task_log-*?h=index")
  for index in ${indices}
    do
      echo -e "Очищен индекс: ${index} "
      ${exec_cmd} "curl -X DELETE -s ${creds} \${ELASTICSEARCH_HOST}:\${ELASTICSEARCH_PORT}/${index}" > /dev/null
  done

  echo -e "Накатываем бэкап.. "
  restore_data_cmd="curl -X POST -s ${creds} ${url}/user_script_backup_snap/_restore?wait_for_completion=true \
  -H 'Content-Type: application/json' \
    -d '{
      \"indices\": \"admin_log-*,user_log-*,mobile_log-*,periodic_task_log-*\",
      \"ignore_unavailable\": true,
      \"include_global_state\": true
    }'"
  ${exec_cmd} "${restore_data_cmd}" > /dev/null


  echo -e "Удаление временных файлов бэкапа "
  ${exec_cmd} "curl -X DELETE -s ${creds} ${url}/user_script_backup_snap" > /dev/null
  ${exec_cmd} "curl -X POST -s ${creds} ${url}/_cleanup" > /dev/null
  ${exec_cmd} "curl -X DELETE -s ${creds} ${url}" > /dev/null
  docker-compose -f ${DOCKER_COMPOSE_DIRECTORY_PATH}/docker-compose.standalone.yml exec -T -u root elasticsearch bash -c "rm -r ./data/user_script_backup"

}
echo -e "Начато резервное восстановление системных логов"
wait_for_elastic_to_up && restore_elasticsearch_data
echo -e "Резервное восстановление системных логов завершено"