import configparser
import subprocess
import sys
import time
from typing import Dict

COMPOSE_PATH = "."


def patch_elastic_compose_file(path_to_file):
    """Добавляет в compose файл настройку необходимую для работы механизма Elasticsearch snapshot"""

    # Актуально для версии ФМП 23.05, в 23.12 данная настройка есть по умолчанию

    with open(f"{path_to_file}/docker-compose.standalone.yml", 'r') as f:
        patch_line = "      - path.repo=/usr/share/elasticsearch\n"
        file_lines = f.readlines()
        if patch_line not in file_lines:
            file_lines.insert(96, patch_line)
            with open(f"{path_to_file}/docker-compose.standalone.yml", 'w') as new_f:
                new_f.writelines(file_lines)

            command = f"docker-compose -f {path_to_file}/docker-compose.standalone.yml up -d --build elasticsearch"
            subprocess.run(command.split())
            print("Ждем перезапуска сервиса ElasticSearch")
            time.sleep(30)


def make_backup_data(docker_compose_file_path: str, compose_project_name: str, data_folder: str):
    """Создание бэкапа для Эластика"""

    command_args = [
        "bash",
        "create_elastic_backup_standalone.sh",
        docker_compose_file_path,
        compose_project_name,
        data_folder
    ]
    return subprocess.run(command_args)


def restore_backup_data(docker_compose_file_path: str, compose_project_name: str, data_folder: str):
    """Восстановление бэкапа логов эластика"""

    command_args = [
        "bash",
        "restore_elastic_backup_standalone.sh",
        docker_compose_file_path,
        compose_project_name,
        data_folder
    ]
    return subprocess.run(command_args)


def get_environment_variables(file_path: str) -> Dict[str, str]:
    with open(file_path) as f:
        file_data: str = f.read()
    if not file_data.startswith('['):
        file_data = f"[generic]\n{file_data}"
    config = configparser.ConfigParser()
    config.read_string(file_data)
    env_variables = {}
    for section in config._sections:
        env_variables.update(config._sections[section])
    return env_variables


def make_backup(backup_folder: str, old_compose_file: str = None):
    if old_compose_file:  # если есть такая переменная значит фмп в процессе обновления
        # и надо брать данные от исходной версии
        environment_variables = get_environment_variables(f"{old_compose_file}/.env")
        patch_elastic_compose_file(old_compose_file)
        make_backup_data(old_compose_file, environment_variables["compose_project_name"], backup_folder)
    else:
        environment_variables = get_environment_variables(f"{COMPOSE_PATH}/.env")
        patch_elastic_compose_file(COMPOSE_PATH)
        make_backup_data(COMPOSE_PATH, environment_variables["compose_project_name"], backup_folder)


def restore_backup(backup_folder: str):
    environment_variables = get_environment_variables(f"{COMPOSE_PATH}/.env")
    patch_elastic_compose_file(COMPOSE_PATH)
    restore_backup_data(COMPOSE_PATH, environment_variables["compose_project_name"], backup_folder)


def show_help():
    msg = 'Данный скрипт принимает в качестве аргументов следующие команды: \n' \
          'backup - Создание бэкапа логов системы \n' \
          'restore - Восстановление логов системы из бэкапа \n' \
          'Пример использования - python3 elasticsearch_backup.py backup'
    print(msg)


if __name__ == "__main__":
    if len(sys.argv) > 1:  # первый аргумент это указание операции( бэкап, восстановление)
        action = sys.argv[1]
    else:
        action = 'help'
    if len(sys.argv) > 2:  # второй аргумент это путь к рабочей директории (docker-compose,.env, e.t.c)
        path = sys.argv[2]
    else:
        path = '.'
    if len(sys.argv) > 3:  # третий аргумент нужен только для обновления т.к
        # принимает путь до папки с исходной версией фмп(до обновления)
        # для рутинных операций бэкапа и восстановления - не требуется.
        old_version_dir = sys.argv[3]
    else:
        old_version_dir = ''
    if action == 'backup' and path:
        make_backup(path, old_version_dir)
    elif action == 'restore' and path:
        restore_backup(path)
    else:
        show_help()
